/*global atob, unescape, Uint8Array, Blob*/

function setupTestHelpers() {
    jasmine.clock().install();
    this.elements = [];
    this.editors = [];

    this.createElement = function (tag, className, html, dontAppend) {
        var el = document.createElement(tag);
        el.innerHTML = html || '';
        if (className) {
            el.className = className;
        }
        this.elements.push(el);
        if (!dontAppend) {
            document.body.appendChild(el);
        }
        return el;
    };

    this.newMediumEditor = function (selector, options) {
        var editor = new MediumEditor(selector, options);
        this.editors.push(editor);
        return editor;
    };

    this.cleanupTest = function () {
        this.editors.forEach(function (editor) {
            editor.destroy();
        });
        this.elements.forEach(function (element) {
            if (element.parentNode) {
                element.parentNode.removeChild(element);
            }
        });

        jasmine.clock().uninstall();

        delete this.createElement;
        delete this.createMedium;
        delete this.elements;
        delete this.editors;
        delete this.cleanupTest;
    }
}

function isIE9() {
    return navigator.appName.indexOf('Internet Explorer') !== -1 && navigator.appVersion.indexOf("MSIE 9") !== -1;
}

function isIE10() {
    return navigator.appName.indexOf('Internet Explorer') !== -1 && navigator.appVersion.indexOf("MSIE 10") !== -1;
}

function isOldIE() {
    return isIE9() || isIE10();
}

function isIE() {
    return ((navigator.appName === 'Microsoft Internet Explorer') || ((navigator.appName === 'Netscape') && (new RegExp('Trident/.*rv:([0-9]{1,}[.0-9]{0,})').exec(navigator.userAgent) !== null)));
}

// If the browser is Edge, returns the version number as a float, otherwise returns 0
function getEdgeVersion() {
    var match = /Edge\/(\d+[,.]\d+)/.exec(navigator.userAgent);
    if (match !== null) {
        return +match[1];
    }
    return 0;
}

function isFirefox() {
    return navigator.userAgent.toLowerCase().indexOf('firefox') !== -1;
}

function isSafari() {
    return navigator.userAgent.toLowerCase().indexOf('safari') !== -1;
}

function dataURItoBlob(dataURI) {
    // convert base64/URLEncoded data component to raw binary data held in a string
    var byteString,
        mimeString,
        ia,
        i;

    if (dataURI.split(',')[0].indexOf('base64') >= 0) {
        byteString = atob(dataURI.split(',')[1]);
    } else {
        byteString = unescape(dataURI.split(',')[1]);
    }

    // separate out the mime component
    mimeString = dataURI.split(',')[0].split(':')[1].split(';')[0];

    // write the bytes of the string to a typed array
    ia = new Uint8Array(byteString.length);
    for (i = 0; i < byteString.length; i += 1) {
        ia[i] = byteString.charCodeAt(i);
    }

    return new Blob([ia], {type: mimeString});
}

// keyCode, ctrlKey, target, relatedTarget, shiftKey, altKey
function fireEvent(element, eventName, options) {
    var evt = prepareEvent(
        element,
        eventName,
        options
    );

    return firePreparedEvent(evt, element, eventName);
}

/**
 * prepareEvent works with firePreparedEvent.
 *
 * It allows test to:
 *     - create the event
 *     - spy a method on this event
 *     - fire the event
 *
 * Example:
 *     var p = document.querySelector('p');
 *     var evt = prepareEvent(p, 'keydown', { keyCode: MediumEditor.util.keyCode.ENTER });
 *     spyOn(evt, 'preventDefault').and.callThrough();
 *     firePreparedEvent(evt, p, 'keydown');
 *     expect(evt.preventDefault).toHaveBeenCalled();
 *
 * You can see a live example for tests related to `disableDoubleReturn`
 */
function prepareEvent (element, eventName, options) {
    var evt;

    options = options || {};

    if (document.createEvent) {
        // dispatch for firefox + others
        evt = document.createEvent('HTMLEvents');
        evt.initEvent(eventName, true, true); // event type,bubbling,cancelable

        evt.currentTarget = options.currentTarget ? options.currentTarget : element;

        if (options.keyCode) {
            evt.keyCode = options.keyCode;
            evt.which = options.keyCode;
        }

        if (options.ctrlKey) {
            evt.ctrlKey = true;
        }

        if (options.metaKey) {
            evt.metaKey = true;
        }

        if (options.target) {
            evt.target = options.target;
        }

        if (options.relatedTarget) {
            evt.relatedTarget = options.relatedTarget;
        }

        if (options.shiftKey) {
            evt.shiftKey = true;
        }

        if (options.altKey) {
          evt.altKey = true;
        }

        if (eventName.indexOf('drag') !== -1 || eventName === 'drop') {
            evt.dataTransfer = {
                dropEffect: ''
            };
            // File API only allows access to 'files' on drop, not on any other event
            if (!isIE9() && eventName === 'drop') {
                var file = dataURItoBlob('data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7');
                if (!file.type) {
                    file.type = 'image/gif';
                }
                evt.dataTransfer.files = [file];
            }
        }
    } else {
        // dispatch for IE
        evt = document.createEventObject();
    }

    return evt;
}

/**
 * @see prepareEvent
 */
function firePreparedEvent (event, element, eventName) {
    if (document.createEvent) {
        return !element.dispatchEvent(event);
    }

    return element.fireEvent('on' + eventName, event);
}

function placeCursorInsideElement(el, index) {
    var selection = window.getSelection(),
        newRange = document.createRange();
    selection.removeAllRanges();
    newRange.setStart(el, index);
    selection.addRange(newRange);
}

function selectElementContents(el, options) {
    options = options || {};

    var range = document.createRange(),
        sel = window.getSelection();
    range.selectNodeContents(el);

    if (options.collapse) {
        range.collapse(options.collapse === true);
    }

    sel.removeAllRanges();
    sel.addRange(range);
}

function selectElementContentsAndFire(el, options) {
    options = options || {};
    selectElementContents(el, options);
    fireEvent(el, options.eventToFire || 'click');
    if (options.testDelay !== -1) {
        if (!options.testDelay) {
            jasmine.clock().tick(1);
        } else {
            jasmine.clock().tick(options.testDelay);
        }
    }
}

var WORD_PASTE_EXAMPLE = ['<html xmlns:o="urn:schemas-microsoft-com:office:office"',
'xmlns:w="urn:schemas-microsoft-com:office:word"',
'xmlns:m="http://schemas.microsoft.com/office/2004/12/omml"',
'xmlns="http://www.w3.org/TR/REC-html40">',
'',
'<head>',
'<meta name=Title content="">',
'<meta name=Keywords content="">',
'<meta http-equiv=Content-Type content="text/html; charset=utf-8">',
'<meta name=ProgId content=Word.Document>',
'<meta name=Generator content="Microsoft Word 15">',
'<meta name=Originator content="Microsoft Word 15">',
'<link rel=File-List',
'href="file://localhost/Users/nate/Library/Group%20Containers/UBF8T346G9.Office/msoclip1/01/clip_filelist.xml">',
'<!--[if gte mso 9]><xml>',
' <o:OfficeDocumentSettings>',
'  <o:AllowPNG/>',
' </o:OfficeDocumentSettings>',
'</xml><![endif]-->',
'<link rel=themeData',
'href="file://localhost/Users/nate/Library/Group%20Containers/UBF8T346G9.Office/msoclip1/01/clip_themedata.thmx">',
'<!--[if gte mso 9]><xml>',
' <w:WordDocument>',
'  <w:View>Normal</w:View>',
'  <w:Zoom>0</w:Zoom>',
'  <w:TrackMoves/>',
'  <w:TrackFormatting/>',
'  <w:PunctuationKerning/>',
'  <w:ValidateAgainstSchemas/>',
'  <w:SaveIfXMLInvalid>false</w:SaveIfXMLInvalid>',
'  <w:IgnoreMixedContent>false</w:IgnoreMixedContent>',
'  <w:AlwaysShowPlaceholderText>false</w:AlwaysShowPlaceholderText>',
'  <w:DoNotPromoteQF/>',
'  <w:LidThemeOther>EN-US</w:LidThemeOther>',
'  <w:LidThemeAsian>JA</w:LidThemeAsian>',
'  <w:LidThemeComplexScript>X-NONE</w:LidThemeComplexScript>',
'  <w:Compatibility>',
'   <w:BreakWrappedTables/>',
'   <w:SnapToGridInCell/>',
'   <w:WrapTextWithPunct/>',
'   <w:UseAsianBreakRules/>',
'   <w:DontGrowAutofit/>',
'   <w:SplitPgBreakAndParaMark/>',
'   <w:EnableOpenTypeKerning/>',
'   <w:DontFlipMirrorIndents/>',
'   <w:OverrideTableStyleHps/>',
'   <w:UseFELayout/>',
'  </w:Compatibility>',
'  <m:mathPr>',
'   <m:mathFont m:val="Cambria Math"/>',
'   <m:brkBin m:val="before"/>',
'   <m:brkBinSub m:val="&#45;-"/>',
'   <m:smallFrac m:val="off"/>',
'   <m:dispDef/>',
'   <m:lMargin m:val="0"/>',
'   <m:rMargin m:val="0"/>',
'   <m:defJc m:val="centerGroup"/>',
'   <m:wrapIndent m:val="1440"/>',
'   <m:intLim m:val="subSup"/>',
'   <m:naryLim m:val="undOvr"/>',
'  </m:mathPr></w:WordDocument>',
'</xml><![endif]--><!--[if gte mso 9]><xml>',
' <w:LatentStyles DefLockedState="false" DefUnhideWhenUsed="false"',
'  DefSemiHidden="false" DefQFormat="false" DefPriority="99"',
'  LatentStyleCount="380">',
'  <w:LsdException Locked="false" Priority="0" QFormat="true" Name="Normal"/>',
'  <w:LsdException Locked="false" Priority="9" QFormat="true" Name="heading 1"/>',
'  <w:LsdException Locked="false" Priority="9" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="heading 2"/>',
'  <w:LsdException Locked="false" Priority="9" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="heading 3"/>',
'  <w:LsdException Locked="false" Priority="9" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="heading 4"/>',
'  <w:LsdException Locked="false" Priority="9" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="heading 5"/>',
'  <w:LsdException Locked="false" Priority="9" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="heading 6"/>',
'  <w:LsdException Locked="false" Priority="9" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="heading 7"/>',
'  <w:LsdException Locked="false" Priority="9" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="heading 8"/>',
'  <w:LsdException Locked="false" Priority="9" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="heading 9"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 5"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 6"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 7"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 8"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index 9"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 1"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 2"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 3"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 4"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 5"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 6"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 7"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 8"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="toc 9"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Normal Indent"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="footnote text"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="annotation text"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="header"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="footer"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="index heading"/>',
'  <w:LsdException Locked="false" Priority="35" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="caption"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="table of figures"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="envelope address"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="envelope return"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="footnote reference"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="annotation reference"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="line number"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="page number"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="endnote reference"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="endnote text"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="table of authorities"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="macro"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="toa heading"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Bullet"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Number"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List 5"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Bullet 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Bullet 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Bullet 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Bullet 5"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Number 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Number 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Number 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Number 5"/>',
'  <w:LsdException Locked="false" Priority="10" QFormat="true" Name="Title"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Closing"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Signature"/>',
'  <w:LsdException Locked="false" Priority="1" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="Default Paragraph Font"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Body Text"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Body Text Indent"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Continue"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Continue 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Continue 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Continue 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="List Continue 5"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Message Header"/>',
'  <w:LsdException Locked="false" Priority="11" QFormat="true" Name="Subtitle"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Salutation"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Date"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Body Text First Indent"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Body Text First Indent 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Heading"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Body Text 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Body Text 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Body Text Indent 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Body Text Indent 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Block Text"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Hyperlink"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="FollowedHyperlink"/>',
'  <w:LsdException Locked="false" Priority="22" QFormat="true" Name="Strong"/>',
'  <w:LsdException Locked="false" Priority="20" QFormat="true" Name="Emphasis"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Document Map"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Plain Text"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="E-mail Signature"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Top of Form"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Bottom of Form"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Normal (Web)"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Acronym"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Address"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Cite"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Code"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Definition"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Keyboard"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Preformatted"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Sample"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Typewriter"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="HTML Variable"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Normal Table"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="annotation subject"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="No List"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Outline List 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Outline List 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Outline List 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Simple 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Simple 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Simple 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Classic 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Classic 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Classic 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Classic 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Colorful 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Colorful 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Colorful 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Columns 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Columns 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Columns 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Columns 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Columns 5"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Grid 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Grid 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Grid 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Grid 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Grid 5"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Grid 6"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Grid 7"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Grid 8"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table List 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table List 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table List 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table List 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table List 5"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table List 6"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table List 7"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table List 8"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table 3D effects 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table 3D effects 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table 3D effects 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Contemporary"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Elegant"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Professional"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Subtle 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Subtle 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Web 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Web 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Web 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Balloon Text"/>',
'  <w:LsdException Locked="false" Priority="39" Name="Table Grid"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Table Theme"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 2"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 3"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 4"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 5"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 6"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 7"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 8"/>',
'  <w:LsdException Locked="false" SemiHidden="true" UnhideWhenUsed="true"',
'   Name="Note Level 9"/>',
'  <w:LsdException Locked="false" SemiHidden="true" Name="Placeholder Text"/>',
'  <w:LsdException Locked="false" Priority="1" QFormat="true" Name="No Spacing"/>',
'  <w:LsdException Locked="false" Priority="60" Name="Light Shading"/>',
'  <w:LsdException Locked="false" Priority="61" Name="Light List"/>',
'  <w:LsdException Locked="false" Priority="62" Name="Light Grid"/>',
'  <w:LsdException Locked="false" Priority="63" Name="Medium Shading 1"/>',
'  <w:LsdException Locked="false" Priority="64" Name="Medium Shading 2"/>',
'  <w:LsdException Locked="false" Priority="65" Name="Medium List 1"/>',
'  <w:LsdException Locked="false" Priority="66" Name="Medium List 2"/>',
'  <w:LsdException Locked="false" Priority="67" Name="Medium Grid 1"/>',
'  <w:LsdException Locked="false" Priority="68" Name="Medium Grid 2"/>',
'  <w:LsdException Locked="false" Priority="69" Name="Medium Grid 3"/>',
'  <w:LsdException Locked="false" Priority="70" Name="Dark List"/>',
'  <w:LsdException Locked="false" Priority="71" Name="Colorful Shading"/>',
'  <w:LsdException Locked="false" Priority="72" Name="Colorful List"/>',
'  <w:LsdException Locked="false" Priority="73" Name="Colorful Grid"/>',
'  <w:LsdException Locked="false" Priority="60" Name="Light Shading Accent 1"/>',
'  <w:LsdException Locked="false" Priority="61" Name="Light List Accent 1"/>',
'  <w:LsdException Locked="false" Priority="62" Name="Light Grid Accent 1"/>',
'  <w:LsdException Locked="false" Priority="63" Name="Medium Shading 1 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="64" Name="Medium Shading 2 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="65" Name="Medium List 1 Accent 1"/>',
'  <w:LsdException Locked="false" SemiHidden="true" Name="Revision"/>',
'  <w:LsdException Locked="false" Priority="34" QFormat="true"',
'   Name="List Paragraph"/>',
'  <w:LsdException Locked="false" Priority="29" QFormat="true" Name="Quote"/>',
'  <w:LsdException Locked="false" Priority="30" QFormat="true"',
'   Name="Intense Quote"/>',
'  <w:LsdException Locked="false" Priority="66" Name="Medium List 2 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="67" Name="Medium Grid 1 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="68" Name="Medium Grid 2 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="69" Name="Medium Grid 3 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="70" Name="Dark List Accent 1"/>',
'  <w:LsdException Locked="false" Priority="71" Name="Colorful Shading Accent 1"/>',
'  <w:LsdException Locked="false" Priority="72" Name="Colorful List Accent 1"/>',
'  <w:LsdException Locked="false" Priority="73" Name="Colorful Grid Accent 1"/>',
'  <w:LsdException Locked="false" Priority="60" Name="Light Shading Accent 2"/>',
'  <w:LsdException Locked="false" Priority="61" Name="Light List Accent 2"/>',
'  <w:LsdException Locked="false" Priority="62" Name="Light Grid Accent 2"/>',
'  <w:LsdException Locked="false" Priority="63" Name="Medium Shading 1 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="64" Name="Medium Shading 2 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="65" Name="Medium List 1 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="66" Name="Medium List 2 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="67" Name="Medium Grid 1 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="68" Name="Medium Grid 2 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="69" Name="Medium Grid 3 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="70" Name="Dark List Accent 2"/>',
'  <w:LsdException Locked="false" Priority="71" Name="Colorful Shading Accent 2"/>',
'  <w:LsdException Locked="false" Priority="72" Name="Colorful List Accent 2"/>',
'  <w:LsdException Locked="false" Priority="73" Name="Colorful Grid Accent 2"/>',
'  <w:LsdException Locked="false" Priority="60" Name="Light Shading Accent 3"/>',
'  <w:LsdException Locked="false" Priority="61" Name="Light List Accent 3"/>',
'  <w:LsdException Locked="false" Priority="62" Name="Light Grid Accent 3"/>',
'  <w:LsdException Locked="false" Priority="63" Name="Medium Shading 1 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="64" Name="Medium Shading 2 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="65" Name="Medium List 1 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="66" Name="Medium List 2 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="67" Name="Medium Grid 1 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="68" Name="Medium Grid 2 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="69" Name="Medium Grid 3 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="70" Name="Dark List Accent 3"/>',
'  <w:LsdException Locked="false" Priority="71" Name="Colorful Shading Accent 3"/>',
'  <w:LsdException Locked="false" Priority="72" Name="Colorful List Accent 3"/>',
'  <w:LsdException Locked="false" Priority="73" Name="Colorful Grid Accent 3"/>',
'  <w:LsdException Locked="false" Priority="60" Name="Light Shading Accent 4"/>',
'  <w:LsdException Locked="false" Priority="61" Name="Light List Accent 4"/>',
'  <w:LsdException Locked="false" Priority="62" Name="Light Grid Accent 4"/>',
'  <w:LsdException Locked="false" Priority="63" Name="Medium Shading 1 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="64" Name="Medium Shading 2 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="65" Name="Medium List 1 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="66" Name="Medium List 2 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="67" Name="Medium Grid 1 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="68" Name="Medium Grid 2 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="69" Name="Medium Grid 3 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="70" Name="Dark List Accent 4"/>',
'  <w:LsdException Locked="false" Priority="71" Name="Colorful Shading Accent 4"/>',
'  <w:LsdException Locked="false" Priority="72" Name="Colorful List Accent 4"/>',
'  <w:LsdException Locked="false" Priority="73" Name="Colorful Grid Accent 4"/>',
'  <w:LsdException Locked="false" Priority="60" Name="Light Shading Accent 5"/>',
'  <w:LsdException Locked="false" Priority="61" Name="Light List Accent 5"/>',
'  <w:LsdException Locked="false" Priority="62" Name="Light Grid Accent 5"/>',
'  <w:LsdException Locked="false" Priority="63" Name="Medium Shading 1 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="64" Name="Medium Shading 2 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="65" Name="Medium List 1 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="66" Name="Medium List 2 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="67" Name="Medium Grid 1 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="68" Name="Medium Grid 2 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="69" Name="Medium Grid 3 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="70" Name="Dark List Accent 5"/>',
'  <w:LsdException Locked="false" Priority="71" Name="Colorful Shading Accent 5"/>',
'  <w:LsdException Locked="false" Priority="72" Name="Colorful List Accent 5"/>',
'  <w:LsdException Locked="false" Priority="73" Name="Colorful Grid Accent 5"/>',
'  <w:LsdException Locked="false" Priority="60" Name="Light Shading Accent 6"/>',
'  <w:LsdException Locked="false" Priority="61" Name="Light List Accent 6"/>',
'  <w:LsdException Locked="false" Priority="62" Name="Light Grid Accent 6"/>',
'  <w:LsdException Locked="false" Priority="63" Name="Medium Shading 1 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="64" Name="Medium Shading 2 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="65" Name="Medium List 1 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="66" Name="Medium List 2 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="67" Name="Medium Grid 1 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="68" Name="Medium Grid 2 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="69" Name="Medium Grid 3 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="70" Name="Dark List Accent 6"/>',
'  <w:LsdException Locked="false" Priority="71" Name="Colorful Shading Accent 6"/>',
'  <w:LsdException Locked="false" Priority="72" Name="Colorful List Accent 6"/>',
'  <w:LsdException Locked="false" Priority="73" Name="Colorful Grid Accent 6"/>',
'  <w:LsdException Locked="false" Priority="19" QFormat="true"',
'   Name="Subtle Emphasis"/>',
'  <w:LsdException Locked="false" Priority="21" QFormat="true"',
'   Name="Intense Emphasis"/>',
'  <w:LsdException Locked="false" Priority="31" QFormat="true"',
'   Name="Subtle Reference"/>',
'  <w:LsdException Locked="false" Priority="32" QFormat="true"',
'   Name="Intense Reference"/>',
'  <w:LsdException Locked="false" Priority="33" QFormat="true" Name="Book Title"/>',
'  <w:LsdException Locked="false" Priority="37" SemiHidden="true"',
'   UnhideWhenUsed="true" Name="Bibliography"/>',
'  <w:LsdException Locked="false" Priority="39" SemiHidden="true"',
'   UnhideWhenUsed="true" QFormat="true" Name="TOC Heading"/>',
'  <w:LsdException Locked="false" Priority="41" Name="Plain Table 1"/>',
'  <w:LsdException Locked="false" Priority="42" Name="Plain Table 2"/>',
'  <w:LsdException Locked="false" Priority="43" Name="Plain Table 3"/>',
'  <w:LsdException Locked="false" Priority="44" Name="Plain Table 4"/>',
'  <w:LsdException Locked="false" Priority="45" Name="Plain Table 5"/>',
'  <w:LsdException Locked="false" Priority="40" Name="Grid Table Light"/>',
'  <w:LsdException Locked="false" Priority="46" Name="Grid Table 1 Light"/>',
'  <w:LsdException Locked="false" Priority="47" Name="Grid Table 2"/>',
'  <w:LsdException Locked="false" Priority="48" Name="Grid Table 3"/>',
'  <w:LsdException Locked="false" Priority="49" Name="Grid Table 4"/>',
'  <w:LsdException Locked="false" Priority="50" Name="Grid Table 5 Dark"/>',
'  <w:LsdException Locked="false" Priority="51" Name="Grid Table 6 Colorful"/>',
'  <w:LsdException Locked="false" Priority="52" Name="Grid Table 7 Colorful"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="Grid Table 1 Light Accent 1"/>',
'  <w:LsdException Locked="false" Priority="47" Name="Grid Table 2 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="48" Name="Grid Table 3 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="49" Name="Grid Table 4 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="50" Name="Grid Table 5 Dark Accent 1"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="Grid Table 6 Colorful Accent 1"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="Grid Table 7 Colorful Accent 1"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="Grid Table 1 Light Accent 2"/>',
'  <w:LsdException Locked="false" Priority="47" Name="Grid Table 2 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="48" Name="Grid Table 3 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="49" Name="Grid Table 4 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="50" Name="Grid Table 5 Dark Accent 2"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="Grid Table 6 Colorful Accent 2"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="Grid Table 7 Colorful Accent 2"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="Grid Table 1 Light Accent 3"/>',
'  <w:LsdException Locked="false" Priority="47" Name="Grid Table 2 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="48" Name="Grid Table 3 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="49" Name="Grid Table 4 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="50" Name="Grid Table 5 Dark Accent 3"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="Grid Table 6 Colorful Accent 3"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="Grid Table 7 Colorful Accent 3"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="Grid Table 1 Light Accent 4"/>',
'  <w:LsdException Locked="false" Priority="47" Name="Grid Table 2 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="48" Name="Grid Table 3 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="49" Name="Grid Table 4 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="50" Name="Grid Table 5 Dark Accent 4"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="Grid Table 6 Colorful Accent 4"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="Grid Table 7 Colorful Accent 4"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="Grid Table 1 Light Accent 5"/>',
'  <w:LsdException Locked="false" Priority="47" Name="Grid Table 2 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="48" Name="Grid Table 3 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="49" Name="Grid Table 4 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="50" Name="Grid Table 5 Dark Accent 5"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="Grid Table 6 Colorful Accent 5"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="Grid Table 7 Colorful Accent 5"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="Grid Table 1 Light Accent 6"/>',
'  <w:LsdException Locked="false" Priority="47" Name="Grid Table 2 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="48" Name="Grid Table 3 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="49" Name="Grid Table 4 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="50" Name="Grid Table 5 Dark Accent 6"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="Grid Table 6 Colorful Accent 6"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="Grid Table 7 Colorful Accent 6"/>',
'  <w:LsdException Locked="false" Priority="46" Name="List Table 1 Light"/>',
'  <w:LsdException Locked="false" Priority="47" Name="List Table 2"/>',
'  <w:LsdException Locked="false" Priority="48" Name="List Table 3"/>',
'  <w:LsdException Locked="false" Priority="49" Name="List Table 4"/>',
'  <w:LsdException Locked="false" Priority="50" Name="List Table 5 Dark"/>',
'  <w:LsdException Locked="false" Priority="51" Name="List Table 6 Colorful"/>',
'  <w:LsdException Locked="false" Priority="52" Name="List Table 7 Colorful"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="List Table 1 Light Accent 1"/>',
'  <w:LsdException Locked="false" Priority="47" Name="List Table 2 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="48" Name="List Table 3 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="49" Name="List Table 4 Accent 1"/>',
'  <w:LsdException Locked="false" Priority="50" Name="List Table 5 Dark Accent 1"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="List Table 6 Colorful Accent 1"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="List Table 7 Colorful Accent 1"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="List Table 1 Light Accent 2"/>',
'  <w:LsdException Locked="false" Priority="47" Name="List Table 2 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="48" Name="List Table 3 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="49" Name="List Table 4 Accent 2"/>',
'  <w:LsdException Locked="false" Priority="50" Name="List Table 5 Dark Accent 2"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="List Table 6 Colorful Accent 2"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="List Table 7 Colorful Accent 2"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="List Table 1 Light Accent 3"/>',
'  <w:LsdException Locked="false" Priority="47" Name="List Table 2 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="48" Name="List Table 3 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="49" Name="List Table 4 Accent 3"/>',
'  <w:LsdException Locked="false" Priority="50" Name="List Table 5 Dark Accent 3"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="List Table 6 Colorful Accent 3"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="List Table 7 Colorful Accent 3"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="List Table 1 Light Accent 4"/>',
'  <w:LsdException Locked="false" Priority="47" Name="List Table 2 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="48" Name="List Table 3 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="49" Name="List Table 4 Accent 4"/>',
'  <w:LsdException Locked="false" Priority="50" Name="List Table 5 Dark Accent 4"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="List Table 6 Colorful Accent 4"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="List Table 7 Colorful Accent 4"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="List Table 1 Light Accent 5"/>',
'  <w:LsdException Locked="false" Priority="47" Name="List Table 2 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="48" Name="List Table 3 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="49" Name="List Table 4 Accent 5"/>',
'  <w:LsdException Locked="false" Priority="50" Name="List Table 5 Dark Accent 5"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="List Table 6 Colorful Accent 5"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="List Table 7 Colorful Accent 5"/>',
'  <w:LsdException Locked="false" Priority="46"',
'   Name="List Table 1 Light Accent 6"/>',
'  <w:LsdException Locked="false" Priority="47" Name="List Table 2 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="48" Name="List Table 3 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="49" Name="List Table 4 Accent 6"/>',
'  <w:LsdException Locked="false" Priority="50" Name="List Table 5 Dark Accent 6"/>',
'  <w:LsdException Locked="false" Priority="51"',
'   Name="List Table 6 Colorful Accent 6"/>',
'  <w:LsdException Locked="false" Priority="52"',
'   Name="List Table 7 Colorful Accent 6"/>',
' </w:LatentStyles>',
'</xml><![endif]-->',
'<style>',
'<!--',
' /* Font Definitions */',
'@font-face',
'	{font-family:"Courier New";',
'	panose-1:2 7 3 9 2 2 5 2 4 4;',
'	mso-font-charset:0;',
'	mso-generic-font-family:auto;',
'	mso-font-pitch:variable;',
'	mso-font-signature:-536859905 -1073711037 9 0 511 0;}',
'@font-face',
'	{font-family:Wingdings;',
'	panose-1:5 0 0 0 0 0 0 0 0 0;',
'	mso-font-charset:2;',
'	mso-generic-font-family:auto;',
'	mso-font-pitch:variable;',
'	mso-font-signature:0 268435456 0 0 -2147483648 0;}',
'@font-face',
'	{font-family:"ＭＳ 明朝";',
'	mso-font-charset:128;',
'	mso-generic-font-family:auto;',
'	mso-font-pitch:variable;',
'	mso-font-signature:-536870145 1791491579 134217746 0 131231 0;}',
'@font-face',
'	{font-family:"Cambria Math";',
'	panose-1:2 4 5 3 5 4 6 3 2 4;',
'	mso-font-charset:0;',
'	mso-generic-font-family:auto;',
'	mso-font-pitch:variable;',
'	mso-font-signature:-536870145 1107305727 0 0 415 0;}',
'@font-face',
'	{font-family:Cambria;',
'	panose-1:2 4 5 3 5 4 6 3 2 4;',
'	mso-font-charset:0;',
'	mso-generic-font-family:auto;',
'	mso-font-pitch:variable;',
'	mso-font-signature:-536870145 1073743103 0 0 415 0;}',
'@font-face',
'	{font-family:"Comic Sans MS";',
'	panose-1:3 15 7 2 3 3 2 2 2 4;',
'	mso-font-charset:0;',
'	mso-generic-font-family:auto;',
'	mso-font-pitch:variable;',
'	mso-font-signature:647 0 0 0 159 0;}',
' /* Style Definitions */',
'p.MsoNormal, li.MsoNormal, div.MsoNormal',
'	{mso-style-unhide:no;',
'	mso-style-qformat:yes;',
'	mso-style-parent:"";',
'	margin:0in;',
'	margin-bottom:.0001pt;',
'	mso-pagination:widow-orphan;',
'	font-size:12.0pt;',
'	font-family:Cambria;',
'	mso-ascii-font-family:Cambria;',
'	mso-ascii-theme-font:minor-latin;',
'	mso-fareast-font-family:"ＭＳ 明朝";',
'	mso-fareast-theme-font:minor-fareast;',
'	mso-hansi-font-family:Cambria;',
'	mso-hansi-theme-font:minor-latin;',
'	mso-bidi-font-family:"Times New Roman";',
'	mso-bidi-theme-font:minor-bidi;}',
'p.MsoListParagraph, li.MsoListParagraph, div.MsoListParagraph',
'	{mso-style-priority:34;',
'	mso-style-unhide:no;',
'	mso-style-qformat:yes;',
'	margin-top:0in;',
'	margin-right:0in;',
'	margin-bottom:0in;',
'	margin-left:.5in;',
'	margin-bottom:.0001pt;',
'	mso-add-space:auto;',
'	mso-pagination:widow-orphan;',
'	font-size:12.0pt;',
'	font-family:Cambria;',
'	mso-ascii-font-family:Cambria;',
'	mso-ascii-theme-font:minor-latin;',
'	mso-fareast-font-family:"ＭＳ 明朝";',
'	mso-fareast-theme-font:minor-fareast;',
'	mso-hansi-font-family:Cambria;',
'	mso-hansi-theme-font:minor-latin;',
'	mso-bidi-font-family:"Times New Roman";',
'	mso-bidi-theme-font:minor-bidi;}',
'p.MsoListParagraphCxSpFirst, li.MsoListParagraphCxSpFirst, div.MsoListParagraphCxSpFirst',
'	{mso-style-priority:34;',
'	mso-style-unhide:no;',
'	mso-style-qformat:yes;',
'	mso-style-type:export-only;',
'	margin-top:0in;',
'	margin-right:0in;',
'	margin-bottom:0in;',
'	margin-left:.5in;',
'	margin-bottom:.0001pt;',
'	mso-add-space:auto;',
'	mso-pagination:widow-orphan;',
'	font-size:12.0pt;',
'	font-family:Cambria;',
'	mso-ascii-font-family:Cambria;',
'	mso-ascii-theme-font:minor-latin;',
'	mso-fareast-font-family:"ＭＳ 明朝";',
'	mso-fareast-theme-font:minor-fareast;',
'	mso-hansi-font-family:Cambria;',
'	mso-hansi-theme-font:minor-latin;',
'	mso-bidi-font-family:"Times New Roman";',
'	mso-bidi-theme-font:minor-bidi;}',
'p.MsoListParagraphCxSpMiddle, li.MsoListParagraphCxSpMiddle, div.MsoListParagraphCxSpMiddle',
'	{mso-style-priority:34;',
'	mso-style-unhide:no;',
'	mso-style-qformat:yes;',
'	mso-style-type:export-only;',
'	margin-top:0in;',
'	margin-right:0in;',
'	margin-bottom:0in;',
'	margin-left:.5in;',
'	margin-bottom:.0001pt;',
'	mso-add-space:auto;',
'	mso-pagination:widow-orphan;',
'	font-size:12.0pt;',
'	font-family:Cambria;',
'	mso-ascii-font-family:Cambria;',
'	mso-ascii-theme-font:minor-latin;',
'	mso-fareast-font-family:"ＭＳ 明朝";',
'	mso-fareast-theme-font:minor-fareast;',
'	mso-hansi-font-family:Cambria;',
'	mso-hansi-theme-font:minor-latin;',
'	mso-bidi-font-family:"Times New Roman";',
'	mso-bidi-theme-font:minor-bidi;}',
'p.MsoListParagraphCxSpLast, li.MsoListParagraphCxSpLast, div.MsoListParagraphCxSpLast',
'	{mso-style-priority:34;',
'	mso-style-unhide:no;',
'	mso-style-qformat:yes;',
'	mso-style-type:export-only;',
'	margin-top:0in;',
'	margin-right:0in;',
'	margin-bottom:0in;',
'	margin-left:.5in;',
'	margin-bottom:.0001pt;',
'	mso-add-space:auto;',
'	mso-pagination:widow-orphan;',
'	font-size:12.0pt;',
'	font-family:Cambria;',
'	mso-ascii-font-family:Cambria;',
'	mso-ascii-theme-font:minor-latin;',
'	mso-fareast-font-family:"ＭＳ 明朝";',
'	mso-fareast-theme-font:minor-fareast;',
'	mso-hansi-font-family:Cambria;',
'	mso-hansi-theme-font:minor-latin;',
'	mso-bidi-font-family:"Times New Roman";',
'	mso-bidi-theme-font:minor-bidi;}',
'.MsoChpDefault',
'	{mso-style-type:export-only;',
'	mso-default-props:yes;',
'	font-family:Cambria;',
'	mso-ascii-font-family:Cambria;',
'	mso-ascii-theme-font:minor-latin;',
'	mso-fareast-font-family:"ＭＳ 明朝";',
'	mso-fareast-theme-font:minor-fareast;',
'	mso-hansi-font-family:Cambria;',
'	mso-hansi-theme-font:minor-latin;',
'	mso-bidi-font-family:"Times New Roman";',
'	mso-bidi-theme-font:minor-bidi;}',
'@page WordSection1',
'	{size:8.5in 11.0in;',
'	margin:1.0in 1.0in 1.0in 1.0in;',
'	mso-header-margin:.5in;',
'	mso-footer-margin:.5in;',
'	mso-paper-source:0;}',
'div.WordSection1',
'	{page:WordSection1;}',
' /* List Definitions */',
'@list l0',
'	{mso-list-id:1751544061;',
'	mso-list-type:hybrid;',
'	mso-list-template-ids:-622833810 67698689 67698691 67698693 67698689 67698691 67698693 67698689 67698691 67698693;}',
'@list l0:level1',
'	{mso-level-number-format:bullet;',
'	mso-level-text:;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:Symbol;}',
'@list l0:level2',
'	{mso-level-number-format:bullet;',
'	mso-level-text:o;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:"Courier New";',
'	mso-bidi-font-family:"Times New Roman";}',
'@list l0:level3',
'	{mso-level-number-format:bullet;',
'	mso-level-text:;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:Wingdings;}',
'@list l0:level4',
'	{mso-level-number-format:bullet;',
'	mso-level-text:;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:Symbol;}',
'@list l0:level5',
'	{mso-level-number-format:bullet;',
'	mso-level-text:o;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:"Courier New";',
'	mso-bidi-font-family:"Times New Roman";}',
'@list l0:level6',
'	{mso-level-number-format:bullet;',
'	mso-level-text:;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:Wingdings;}',
'@list l0:level7',
'	{mso-level-number-format:bullet;',
'	mso-level-text:;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:Symbol;}',
'@list l0:level8',
'	{mso-level-number-format:bullet;',
'	mso-level-text:o;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:"Courier New";',
'	mso-bidi-font-family:"Times New Roman";}',
'@list l0:level9',
'	{mso-level-number-format:bullet;',
'	mso-level-text:;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;',
'	font-family:Wingdings;}',
'@list l1',
'	{mso-list-id:1845972260;',
'	mso-list-type:hybrid;',
'	mso-list-template-ids:1079810306 67698703 67698713 67698715 67698703 67698713 67698715 67698703 67698713 67698715;}',
'@list l1:level1',
'	{mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;}',
'@list l1:level2',
'	{mso-level-number-format:alpha-lower;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;}',
'@list l1:level3',
'	{mso-level-number-format:roman-lower;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:right;',
'	text-indent:-9.0pt;}',
'@list l1:level4',
'	{mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;}',
'@list l1:level5',
'	{mso-level-number-format:alpha-lower;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;}',
'@list l1:level6',
'	{mso-level-number-format:roman-lower;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:right;',
'	text-indent:-9.0pt;}',
'@list l1:level7',
'	{mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;}',
'@list l1:level8',
'	{mso-level-number-format:alpha-lower;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:left;',
'	text-indent:-.25in;}',
'@list l1:level9',
'	{mso-level-number-format:roman-lower;',
'	mso-level-tab-stop:none;',
'	mso-level-number-position:right;',
'	text-indent:-9.0pt;}',
'ol',
'	{margin-bottom:0in;}',
'ul',
'	{margin-bottom:0in;}',
'-->',
'</style>',
'<!--[if gte mso 10]>',
'<style>',
' /* Style Definitions */',
'table.MsoNormalTable',
'	{mso-style-name:"Table Normal";',
'	mso-tstyle-rowband-size:0;',
'	mso-tstyle-colband-size:0;',
'	mso-style-noshow:yes;',
'	mso-style-priority:99;',
'	mso-style-parent:"";',
'	mso-padding-alt:0in 5.4pt 0in 5.4pt;',
'	mso-para-margin:0in;',
'	mso-para-margin-bottom:.0001pt;',
'	mso-pagination:widow-orphan;',
'	font-size:12.0pt;',
'	font-family:Cambria;',
'	mso-ascii-font-family:Cambria;',
'	mso-ascii-theme-font:minor-latin;',
'	mso-hansi-font-family:Cambria;',
'	mso-hansi-theme-font:minor-latin;}',
'</style>',
'<![endif]-->',
'</head>',
'',
'<body bgcolor=white lang=EN-US style=\'tab-interval:.5in\'>',
'<!--StartFragment-->',
'',
'<p class=MsoNormal><span style=\'font-size:14.0pt;font-family:"Comic Sans MS"\'>My',
'complicated <b style=\'mso-bidi-font-weight:normal\'>word document renders</b> <i ',
'style=\'mso-bidi-font-style:normal\'>like this in the card</i> generator.<o:p></o:p></span></p>',
'',
'<p class=MsoNormal><span style=\'font-size:14.0pt;font-family:"Comic Sans MS"\'><o:p>&nbsp;</o:p></span></p>',
'',
'<p class=MsoNormal><span style=\'font-size:24.0pt;font-family:"Comic Sans MS"\'>Test',
'big <span style=\'color:red\'>text</span><o:p></o:p></span></p>',
'',
'<p class=MsoNormal><span style=\'font-size:24.0pt;font-family:"Comic Sans MS"\'><o:p>&nbsp;</o:p></span></p>',
'',
'<p class=MsoNormal><span style=\'font-size:24.0pt;font-family:"Comic Sans MS"\'><o:p>&nbsp;</o:p></span></p>',
'',
'<p class=MsoNormal><span style=\'font-size:18.0pt;font-family:"Comic Sans MS"\'>Testing',
'smaller text and <s>crossed out text.<o:p></o:p></s></span></p>',
'',
'<p class=MsoNormal><s><span style=\'font-size:18.0pt;font-family:"Comic Sans MS"\'><o:p><span ',
' style=\'text-decoration:none\'>&nbsp;</span></o:p></span></s></p>',
'',
'<p class=MsoListParagraphCxSpFirst style=\'text-indent:-.25in;mso-list:l0 level1 lfo1\'><![if !supportLists]><span ',
'style=\'font-size:14.0pt;font-family:Symbol;mso-fareast-font-family:Symbol;',
'mso-bidi-font-family:Symbol\'><span style=\'mso-list:Ignore\'>·<span ',
'style=\'font:7.0pt "Times New Roman"\'>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; </span></span></span><![endif]><span ',
'style=\'font-size:14.0pt;font-family:"Comic Sans MS"\'>Test list<o:p></o:p></span></p>',
'',
'<p class=MsoListParagraphCxSpMiddle style=\'text-indent:-.25in;mso-list:l0 level1 lfo1\'><![if !supportLists]><span ',
'style=\'font-size:14.0pt;font-family:Symbol;mso-fareast-font-family:Symbol;',
'mso-bidi-font-family:Symbol\'><span style=\'mso-list:Ignore\'>·<span ',
'style=\'font:7.0pt "Times New Roman"\'>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; </span></span></span><![endif]><span ',
'style=\'font-size:14.0pt;font-family:"Comic Sans MS"\'>Test<o:p></o:p></span></p>',
'',
'<p class=MsoListParagraphCxSpMiddle style=\'margin-left:1.0in;mso-add-space:',
'auto;text-indent:-.25in;mso-list:l0 level2 lfo1\'><![if !supportLists]><span ',
'style=\'font-size:14.0pt;font-family:"Courier New";mso-fareast-font-family:"Courier New"\'><span ',
'style=\'mso-list:Ignore\'>o<span style=\'font:7.0pt "Times New Roman"\'>&nbsp;&nbsp;',
'</span></span></span><![endif]><span style=\'font-size:14.0pt;font-family:"Comic Sans MS"\'>Test',
'indented<o:p></o:p></span></p>',
'',
'<p class=MsoListParagraphCxSpMiddle style=\'text-indent:-.25in;mso-list:l0 level1 lfo1\'><![if !supportLists]><span ',
'style=\'font-size:14.0pt;font-family:Symbol;mso-fareast-font-family:Symbol;',
'mso-bidi-font-family:Symbol\'><span style=\'mso-list:Ignore\'>·<span ',
'style=\'font:7.0pt "Times New Roman"\'>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; </span></span></span><![endif]><span ',
'style=\'font-size:14.0pt;font-family:"Comic Sans MS"\'>Tes6t<o:p></o:p></span></p>',
'',
'<p class=MsoListParagraphCxSpMiddle><span style=\'font-size:14.0pt;font-family:',
'"Comic Sans MS"\'><o:p>&nbsp;</o:p></span></p>',
'',
'<p class=MsoListParagraphCxSpMiddle style=\'text-indent:-.25in;mso-list:l1 level1 lfo2\'><![if !supportLists]><span ',
'style=\'font-size:14.0pt;font-family:"Comic Sans MS";mso-fareast-font-family:',
'"Comic Sans MS";mso-bidi-font-family:"Comic Sans MS"\'><span style=\'mso-list:',
'Ignore\'>1.<span style=\'font:7.0pt "Times New Roman"\'>&nbsp;&nbsp;&nbsp;&nbsp; </span></span></span><![endif]><span ',
'style=\'font-size:14.0pt;font-family:"Comic Sans MS"\'>tes t test test<o:p></o:p></span></p>',
'',
'<p class=MsoListParagraphCxSpLast style=\'margin-left:1.0in;mso-add-space:auto;',
'text-indent:-.25in;mso-list:l1 level2 lfo2\'><![if !supportLists]><span ',
'style=\'font-size:14.0pt;font-family:"Comic Sans MS";mso-fareast-font-family:',
'"Comic Sans MS";mso-bidi-font-family:"Comic Sans MS"\'><span style=\'mso-list:',
'Ignore\'>a.<span style=\'font:7.0pt "Times New Roman"\'>&nbsp;&nbsp;&nbsp; </span></span></span><![endif]><span ',
'style=\'font-size:14.0pt;font-family:"Comic Sans MS"\'>tes t indented<o:p></o:p></span></p>',
'',
'<p class=MsoNormal><o:p>&nbsp;</o:p></p>',
'',
'<!--EndFragment-->',
'</body>',
'',
'</html>'].join('');
